;;;
;;; libtype.scm - type-related stuff
;;;
;;;   Copyright (c) 2021-2025  Shiro Kawai  <shiro@acm.org>
;;;
;;;   Redistribution and use in source and binary forms, with or without
;;;   modification, are permitted provided that the following conditions
;;;   are met:
;;;
;;;   1. Redistributions of source code must retain the above copyright
;;;      notice, this list of conditions and the following disclaimer.
;;;
;;;   2. Redistributions in binary form must reproduce the above copyright
;;;      notice, this list of conditions and the following disclaimer in the
;;;      documentation and/or other materials provided with the distribution.
;;;
;;;   3. Neither the name of the authors nor the names of its contributors
;;;      may be used to endorse or promote products derived from this
;;;      software without specific prior written permission.
;;;
;;;   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;;;   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;;;   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;;;   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
;;;   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;;;   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
;;;   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;;   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;;   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;;   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;;   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;;;

;; This must be the first form to prevent generation of *.sci file
(declare)

;; In Gauche, types are a data structure that appear in both compile-time
;; and run-time, and describe metalevel properties of run-time data.
;;
;; Gauche has two kinds of types--prescriptive types and descriptive types.
;; Prescriptive types are the types that are used to generate
;; actual data---we also call it classes.  Descriptive types are, otoh,
;; used only to describe the nature of data at the certain point of program
;; execution---for example, you may say the argument must be either <integer>
;; or <boolean>.  The descriptive type can't be used to generate an instance,
;; only to be used to validate and/or infer the actual (generative) type of
;; data.
;;
;; Descriptive types can be constructed by type constructors.  Since the
;; compiler needs to know about types, type constructor expressions
;; are evaluated at the compile time, much like macros.
;;
;; We could've implemented types with macros, but it would be tricky.  Unlike
;; macros, type expression needs to evaluate from inside to outside, just
;; like ordinary expressions.  It's more like compile-time constant folding.
;;
;; Since type handing is deeply intertwined with compiler, we let the compiler
;; recognize type expression specifically, rather than reusing existing
;; evaluation mechanism.  When the compile sees (C x ...) and C has an
;; inlineable binding to an instance of <type-constructor-meta>, it recognizes
;; type expression.
;;
;; We have several kinds of descriptive types.
;;
;;   Constraint types
;;      - We don't have a separate class for this, but this is a kind of
;;        descriptive type that constraints the type the concerned
;;        value can take, e.g. (</> <integer> <string>).   Created by
;;        a type constructor.   Currently, the main use is in `assert-type`.
;;
;;   <native-type>
;;      - A subset of Scheme primitive types that can map onto the native
;;        types.  E.g. <long> is a subset of <integer> but can map onto
;;        the underlying `long` integer value.   Main use is for FFI.
;;
;;   <proxy-type>
;;      - This is a wrapper of an prescriptive types, and as far as type
;;        checking is concerened, it behaves just like the wrapped type.
;;        We need this because classes can be redefined.  When a class
;;        is redefined, a new class instance is created and the class name
;;        is rebound to it.  We want that other aggregate descriptive types
;;        also start referring to the new class, so a descriptive type can't
;;        hold a direct reference to a prescriptive type; instead, we refer
;;        to it through a proxy type.  A proxy type always refer to the
;;        the redefined class.
;;        Proxy types are automatically created and handled under the hood;
;;        users doesn't need to deal with them explicity.


;; This module is not meant to be `use'd.   It is just to hide
;; auxiliary procedures from the rest of the system.  The necessary
;; bindings are injected into 'gauche' module at the initialization time.
(define-module gauche.typeutil)
(select-module gauche.typeutil)
(use util.match)


(inline-stub
 (.include "gauche/priv/configP.h"
           "gauche/priv/classP.h"
           "gauche/priv/nativeP.h"
           "gauche/priv/memoP.h"
           "gauche/priv/typeP.h")

 ;; Metaclass: <type-constructor-meta>
 ;;   Its instance is ScmTypeConstructor.  Provides the following slots.
 ;;   (We don't use generic functions, for they are also called from C runtime).
 ;;
 ;;   initializer :: <descriptive-type> args -> <descriptive-type>
 ;;     Called from a type constructor that fills the descriptive type.
 ;;     It is called from either a compiler or an initcode of precompiled code.
 ;;     Note that a raw class object should never passed in the args---the
 ;;     compiler wraps class objects with a proxy type.  The initializer
 ;;     should raise an error if a class object is given.
 ;;     This procedure must be pure.  We may memoize the constructor result.
 ;;   deconstructor :: <descriptive-type> -> (arg ...)
 ;;     Returns a list of arguments such that when they are passed to the
 ;;     constructor, an equivalent descriptive type is constructed again.
 ;;     This is called from the precompiler to serialize the descriptive type.
 ;;   validator :: <descriptive-type> obj -> <boolean>
 ;;     Returns true iff obj is of type <descriptive-type>.  Called from
 ;;     `of-type?`.
 ;;   subtype? :: <descriptive-type> type -> <boolean>|'super
 ;;     Returns one of three values:  #t if the descriptive type is a subtype
 ;;     of TYPE, which may be a class or another descriptive type.  #f if
 ;;     the descriptive type is definitely not a subtype.  A symbol 'super
 ;;     if you need to ask TYPE.
 ;;     Note that proxy types and native types are already excluded, as well
 ;;     as the reflective case (subtype? x x) and the base cases
 ;;     (subtype? x <top>).
 ;;   supertype? :: <descriptive-type> type -> <boolean>
 ;;     Returns true iff the descriptive type is a supertype of TYPE.
 ;;     Trivial cases are already excluded, esp., TYPE won't be the
 ;;     same kind of <descriptive-type>.

 (define-cclass <type-constructor-meta> :base :private :no-meta
   "ScmTypeConstructor*" "Scm_TypeConstructorMetaClass"
   (c "SCM_CLASS_METACLASS_CPL")
   ((initializer)
    (deconstructor)
    (validator)
    (subtype? :c-name "subtypeP")
    (supertype? :c-name "supertypeP"))
   )

 (define-cfn Scm_TypeConstructorP (klass) ::int
   (return (SCM_ISA klass (& Scm_TypeConstructorMetaClass))))

 ;; The 'name' slot is computed by the initializer.
 ;; The 'constructorArgs' slot is #f by default.  If the instance is
 ;; reconstructed from the precompiled form, however, we delay the actual
 ;; initialization until the type is used.
 (define-ctype ScmDescriptiveType
   ::(.struct ScmDescriptiveTypeRec
              (SCM_INSTANCE_HEADER::||
               name::ScmObj
               constructorArgs::ScmObj)))

 (define-cclass <descriptive-type> :base :private :no-meta
   "ScmDescriptiveType*" "Scm_DescriptiveTypeClass"
   (c "SCM_CLASS_METACLASS_CPL+1")
   ((name))
   (metaclass <type-constructor-meta>)
   (allocator (let* ([z::ScmDescriptiveType*
                      (SCM_NEW_INSTANCE ScmDescriptiveType klass)])
                (cast void initargs)    ;suppress unused warning
                (set! (-> z name) SCM_FALSE)
                (set! (-> z constructorArgs) SCM_FALSE)
                (return (SCM_OBJ z)))))

 ;; We memoize constructed types.  The type is keyed by the constructor class
 ;; and a list of contructor arguments.
 (define-cvar type-table::ScmMemoTable* :static)
 (initcode
  (set! type-table (SCM_MEMO_TABLE (Scm_MakeMemoTable 256 -1 0))))

 (define-cfn lookup-constructed-type (type-ctor::ScmTypeConstructor*
                                      args)
   :static
   (let* ([keys::(.array ScmObj (2))] [r])
     (set! (aref keys 0) (SCM_OBJ type-ctor))
     (set! (aref keys 1) args)
     (set! r (Scm_MemoTableGetv type-table keys 2))
     (if (SCM_UNBOUNDP r)
       (return SCM_FALSE)
       (return r))))

 (define-cfn register-constructed-type (type-ctor::ScmTypeConstructor*
                                        args
                                        type)
   :static
   (let* ([keys::(.array ScmObj (2))])
     (set! (aref keys 0) (SCM_OBJ type-ctor))
     (set! (aref keys 1) args)
     (Scm_MemoTablePutv type-table keys 2 type)
     (return type)))

 (define-cproc %dump-type-table () ::<void> ; for debug
   (Scm__MemoTableDump type-table SCM_CURERR))

 (define-cclass <native-type> :base :no-meta
   "ScmNativeType*" "Scm_NativeTypeClass"
   (c "SCM_CLASS_METACLASS_CPL+1")
   ((name)
    (super)
    (c-type-name :type <const-cstring>)
    (size :type <size_t>)
    (alignment :type <size_t>))
   (printer (Scm_Printf port "#<native-type %A>" (-> (SCM_NATIVE_TYPE obj) name))))

 ;; CPA for native type subclasses
 (define-cvar native-type-cpa::(.array ScmClass* (*)) :static
   #((SCM_CLASS_STATIC_PTR Scm_NativeTypeClass)
     (SCM_CLASS_STATIC_PTR Scm_TopClass)
     NULL))

 (define-cclass <native-pointer> :base :no-meta
   "ScmNativePointer*" "Scm_NativePointerClass"
   (c "native_type_cpa")
   ((pointee-type :type <native-type> :c-name "pointee_type"))
   (printer (Scm_Printf port "#<native-pointer %A>" (-> (& (-> (SCM_NATIVE_POINTER obj) common)) name)))
   (comparer (c Scm_ObjectCompare)))

 (define-cclass <native-function> :base :no-meta
   "ScmNativeFunction*" "Scm_NativeFunctionClass"
   (c "native_type_cpa")
   ((return-type :type <native-type> :c-name "return_type")
    (arg-types :c-name "arg_types")
    (varargs :type <boolean>))
   (printer (Scm_Printf port "#<native-function%A>" (-> (& (-> (SCM_NATIVE_FUNCTION obj) common)) name)))
   (comparer (c Scm_ObjectCompare)))

 (define-cclass <native-array> :base :no-meta
   "ScmNativeArray*" "Scm_NativeArrayClass"
   (c "native_type_cpa")
   ((element-type :type <native-type> :c-name "element_type")
    (dimensions))
   (printer (Scm_Printf port "#<native-array %A>" (-> (& (-> (SCM_NATIVE_ARRAY obj) common)) name)))
   (comparer (c Scm_ObjectCompare)))

 (define-cclass <native-struct> :base :no-meta
   "ScmNativeStruct*" "Scm_NativeStructClass"
   (c "native_type_cpa")
   ((tag)
    (fields))
   (printer (Scm_Printf port "#<native-struct %A>" (-> (& (-> (SCM_NATIVE_STRUCT obj) common)) name)))
   (comparer (c Scm_ObjectCompare)))

 (define-cclass <native-union> :base :no-meta
   "ScmNativeUnion*" "Scm_NativeUnionClass"
   (c "native_type_cpa")
   ((tag)
    (fields))
   (printer (Scm_Printf port "#<native-union %A>" (-> (& (-> (SCM_NATIVE_UNION obj) common)) name)))
   (comparer (c Scm_ObjectCompare)))
 )

(define-method initialize ((c <type-constructor-meta>) initargs)
  (next-method)
  (unless (slot-bound? c 'subtype?)
    (slot-set! c 'subtype? (^[type super] #f)))
  (unless (slot-bound? c 'supertype?)
    (slot-set! c 'supertype? (^[type sub] #f))))

;; Returns a class that's the basis of the native type
(inline-stub
 (define-cfn Scm_NativeTypeBaseClass (np::ScmNativeType*)
   (let* ([t::ScmNativeType* np])
     (loop
      (let* ([sup (-> t super)])
        (cond [(SCM_CLASSP sup) (return sup)]
              [(SCM_NATIVE_TYPE_P sup) (set! t (SCM_NATIVE_TYPE sup))]
              [else (Scm_Error "[internal] Bad inheritance setup: %S" np)])))))
 )

;;;
;;; type? obj
;;;

;; Returns obj if it's either a class metaobject, a descriptive type,
;; a native type, or a proxy type.  It's the same as (is-a? obj <type>)
;; but this is more concise and clear.
(define-cproc type? (obj) ::<boolean> :constant
  (return (SCM_ISA obj SCM_CLASS_TYPE)))

;;;
;;; of-type? obj type
;;;

(inline-stub
 ;;  This may push C continuations on VM, so must be called on VM.
 (define-cfn Scm_VMOfType (obj type)
   (cond [(SCM_PROXY_TYPE_P type)
          (return (Scm_VMIsA obj (Scm_ProxyTypeRef (SCM_PROXY_TYPE type))))]
         [(SCM_DESCRIPTIVE_TYPE_P type)
          (let* ([k::ScmClass* (SCM_CLASS_OF type)])
            (SCM_ASSERT (SCM_TYPE_CONSTRUCTOR_META_P k))
            (return (Scm_VMApply2 (-> (SCM_TYPE_CONSTRUCTOR_META k) validator)
                                  type obj)))]
         [(SCM_NATIVE_TYPE_P type)
          (return (SCM_MAKE_BOOL
                   (funcall (-> (SCM_NATIVE_TYPE type) c-of-type) obj)))]
         [(SCM_CLASSP type)
          (return (Scm_VMIsA obj (SCM_CLASS type)))]
         [else
          (Scm_Error "Second argument of of-type? must be a type, but got: %S"
                     type)]))

 (define-cproc of-type? (obj type) Scm_VMOfType)
 )

;;;
;;; subtype? type-or-class type-or-class
;;;

(define-cfn delegate-to-super (sub super) :static
  (if (SCM_DESCRIPTIVE_TYPE_P super)
    (let* ([k::ScmClass* (Scm_ClassOf super)])
      (SCM_ASSERT (SCM_TYPE_CONSTRUCTOR_META_P k))
      (return (Scm_VMApply2 (-> (SCM_TYPE_CONSTRUCTOR_META k) supertypeP)
                            super sub)))
    (return SCM_FALSE)))

(define-cproc subtype? (sub super)
  (loop
   (cond
    ;; Strip proxy types first
    [(SCM_PROXY_TYPE_P sub)
     (set! sub (SCM_OBJ (Scm_ProxyTypeRef (SCM_PROXY_TYPE sub))))]
    [(SCM_PROXY_TYPE_P super)
     (set! super (SCM_OBJ (Scm_ProxyTypeRef (SCM_PROXY_TYPE super))))]
    ;; Filter out the trivial cases
    [(SCM_EQ super (SCM_OBJ SCM_CLASS_TOP)) (return SCM_TRUE)]
    [(SCM_EQ sub (SCM_OBJ SCM_CLASS_BOTTOM)) (return SCM_TRUE)]
    [(SCM_EQ super sub) (return SCM_TRUE)]
    ;; Native types can be a subtype of a class.  Classes never be a subtype
    ;; of a native type (except <bottom>).
    ;; Native types can form single inheritance chain.
    [(and (SCM_NATIVE_TYPE_P sub)
          (SCM_CLASSP super))
     ;; we fall back to class vs class comparison
     (let* ([klass (Scm_NativeTypeBaseClass (SCM_NATIVE_TYPE sub))])
       (set! sub klass))]        ; retry
    [(and (SCM_NATIVE_TYPE_P sub)
          (SCM_NATIVE_TYPE_P super))
     ;; we have single inheritance between native types
     (set! sub (-> (SCM_NATIVE_TYPE sub) super))
     (loop
      (cond [(SCM_EQ sub super) (return SCM_TRUE)]
            [(SCM_NATIVE_TYPE_P sub)
             (set! sub (-> (SCM_NATIVE_TYPE sub) super))] ;retry
            [else (return SCM_FALSE)]))]
    ;; the case of (subtype? <native-type>  <descriptive-type>) is handled
    ;; later
    [(SCM_NATIVE_TYPE_P super) (return SCM_FALSE)]
    ;; Both are classes, we can use subclass?
    [(and (SCM_CLASSP sub) (SCM_CLASSP super))
     (return (SCM_MAKE_BOOL (Scm_SubclassP (SCM_CLASS sub) (SCM_CLASS super))))]
    ;; Delegate descriptive types to its handlers.
    [(SCM_DESCRIPTIVE_TYPE_P sub)
     (let* ([k::ScmClass* (Scm_ClassOf sub)])
       (SCM_ASSERT (SCM_TYPE_CONSTRUCTOR_META_P k))
       (let1/cps r (Scm_VMApply2 (-> (SCM_TYPE_CONSTRUCTOR_META k) subtypeP)
                                 sub super)
         [sub super]
         (cond [(or (SCM_FALSEP r) (SCM_EQ r SCM_TRUE)) (return r)]
               [(SCM_EQ r 'super) (return (delegate-to-super sub super))]
               [else
                (Scm_Error "subtype? handler of %S returned invalid value: %S"
                           r)])))]
    [else (return (delegate-to-super sub super))])))

;;;
;;; Descriptive type infrastructure
;;;

(define-method allocate-instance ((t <descriptive-type>) initargs)
  (error "Abstract type instance cannot instantiate a concrete object:" t))

(define-method write-object ((t <descriptive-type>) port)
  (format port "#~a" (~ t'name)))

;; Equality is used when consolidate literals.  It's not lightweight
;; (it calls deconstructor, which allocates).
(define-method object-equal? ((x <descriptive-type>) (y <descriptive-type>))
  (and (equal? (class-of x) (class-of y))
       (equal? (deconstruct-type x) (deconstruct-type y))))

;; This can also be called from initialization of precompiled code to recover
;; descripitve type instance.
(inline-stub
 (define-cfn Scm_ConstructType (ctor args)
   (SCM_ASSERT (!= ctor NULL))
   (unless (Scm_TypeConstructorP ctor)
     (SCM_TYPE_ERROR ctor "<type-constructor-meta>"))
   (let* ([ct::ScmTypeConstructor* (cast ScmTypeConstructor* ctor)]
          [type (lookup-constructed-type ct args)])
     (unless (SCM_FALSEP type) (return type))
     (set! type (Scm_NewInstance (SCM_CLASS ct) (sizeof ScmDescriptiveType)))
     (Scm_ApplyRec2 (-> ct initializer) type args)
     (return (register-constructed-type ct args type))))
 )
;; Public interface to construct a descriptive type.
(define-cproc construct-type (meta args) Scm_ConstructType)

;; Internal API, required to precompile descriptive type constant
(define-method deconstruct-type ((t <descriptive-type>))
  ((~ (class-of t)'deconstructor) t))


;;;
;;; Proxy type API.
;;;

;; These are available in gauche.internal

(define-cproc wrap-with-proxy-type (id gloc)
  (unless (SCM_IDENTIFIERP id)
    (SCM_TYPE_ERROR id "identifier"))
  (unless (SCM_GLOCP gloc)
    (SCM_TYPE_ERROR gloc "gloc"))
  (return (Scm_MakeProxyType (SCM_IDENTIFIER id) (SCM_GLOC gloc))))

(define-cproc proxy-type-ref (type)
  (unless (SCM_PROXY_TYPE_P type)
    (SCM_TYPE_ERROR type "proxy-type"))
  (return (SCM_OBJ (Scm_ProxyTypeRef (SCM_PROXY_TYPE type)))))

(define-cproc proxy-type-id (type)
  (unless (SCM_PROXY_TYPE_P type)
    (SCM_TYPE_ERROR type "proxy-type"))
  (return (Scm_ProxyTypeId (SCM_PROXY_TYPE type))))

;; Internal.  Recover type from the serialized type name for precompiled subr.
;; KLUDGE: 'module' here is the module where the subr is defined, but the type
;; itself isn't necessarily visible from that module.  A typical case is the
;; subr defined in 'scheme' module---the types are obviously not visible from
;; vanilla scheme.  For now, we try the given module, then try #<module gauche>.
(define-cproc %lookup-type (mod::<module> type-name::<symbol>)
  (let* ([g::ScmGloc* (Scm_FindBinding mod type-name 0)])
    (when (== g NULL)
      (set! g (Scm_FindBinding (Scm_GaucheModule) type-name 0)))
    (when (== g NULL)
      (return SCM_FALSE))
    (let* ([val (Scm_GlocGetValue g)])
      (cond [(SCM_CLASSP val)
             (let* ([id (Scm_MakeIdentifier (SCM_OBJ type-name) mod SCM_NIL)])
               (return (Scm_MakeProxyType (SCM_IDENTIFIER id) g)))]
            [(SCM_ISA val SCM_CLASS_TYPE) (return val)]
            [else (return SCM_FALSE)]))))

(define (%type-name->type module type-name)
  (define (maybe-type y)
    (let1 s (symbol->string y)
      (and (eqv? (string-ref s (- (string-length s) 1)) #\?)
           (string->symbol (substring s 0 (- (string-length s) 1))))))
  (if-let1 m (maybe-type type-name)
    (and-let1 mt (%lookup-type module m)
      (construct-type <?> (list mt)))
    (%lookup-type module type-name)))

;;;
;;; Utilities
;;;

(define (join-class-names classes)
  (string-join (map (^k (x->string
                         (cond [(is-a? k <class>) (class-name k)]
                               [(is-a? k <descriptive-type>) (~ k'name)]
                               [(is-a? k <native-type>) (~ k'name)]
                               [(is-a? k <proxy-type>)
                                (~ (proxy-type-id k) 'name)]
                               [else k])))
                    classes)
               " " 'prefix))

(define (make-compound-type-name op-name classes)
  ($ string->symbol
     $ string-append "<" (x->string op-name) (join-class-names classes) ">"))

(define (make-min-max-len-type-name op-name classes min max)
  ($ string->symbol
     $ string-append "<" (x->string op-name) (join-class-names classes)
     (if min
       (if max
         (if (= min max)
           (format " ~d" min)
           (format " ~d..~d" min max))
         (format " ~d.." min))
       (if max
         (format " ..~d" max)
         ""))
     ">"))

;;;
;;; Class: <^>  (maybe we want to name it <λ>)
;;;   Creates a procedure type.
;;;   The signature can be specified as
;;;
;;;       <argtype1> <argtype2> ... -> <rettype1> <rettype2> ...
;;;
;;;   Argument types and/or return types can be also a single symbol *,
;;;   indicating arbitrary number of args/values.   That is, any procedure
;;;   can be of type * -> *.
;;;
;;;   TODO: How to type optional and keyword arguments?
;;;


(define (init-^ type init-args)
  (define (scan-args xs as)
    ;; NB: We want to allow a right arrow (U+2192) in place of ->.
    ;; The code would be messy though, if we want to support 'none' encoding as
    ;; well.  Think about it after we drop 'none' support.
    (match xs
      [() (error "Missing '->' in the procedure type constructor arguments:"
                 init-args)]
      [('-> . xs) (scan-results xs (reverse as) '())]
      [('* '-> . xs) (scan-results xs (reverse as '(*)) '())]
      [_
       (if (is-a? (car xs) <type>)
         (scan-args (cdr xs) (cons (car xs) as))
         (error "Non-type argument in the procedure type constructor:"
                (car xs)))]))
  (define (scan-results xs args rs)
    (match xs
      [() (values args (reverse rs))]
      [('*) (values args (reverse rs '(*)))]
      [(x . xs)
       (if (is-a? x <type>)
         (scan-results xs args (cons x rs))
         (error "Non-class argument in the procedure type constructor:" x))]
      [_ (error "Invalid arguments:" xs)]))

  (receive (args results) (scan-args init-args '())
    (slot-set! type 'name      (make-compound-type-name '^ init-args))
    (slot-set! type 'arguments (construct-type <Tuple> args))
    (slot-set! type 'results   (construct-type <Tuple> results))))

(define (deconstruct-^ type)
  (append (~ type'arguments'elements)
          (if (~ type'arguments'allow-rest?) '(*) '())
          '(->)
          (~ type'results'elements)
          (if (~ type'results'allow-rest?) '(*) '())))

(define (validate-^ type obj)
  (and-let1 otype (compute-procedure-type obj)
    (subtype? type otype)))

(define (subtype-^ type super)
  (if (is-a? super <^>)
    (and (subtype? (~ type'arguments) (~ super'arguments))
         (subtype? (~ super'results) (~ type'results)))
    'super))

;; Internal API - called from procedure-type (libproc)
;; Reconstruct #<^ ...> type from a serialized type info encoded in a vector.
(define (reconstruct-procedure-type proc encoded-type)
  (if (and (vector? encoded-type)
           (>= (vector-length encoded-type) 3)
           (= (vector-ref encoded-type 0) 1))
    (let* ([module-name (vector-ref encoded-type 1)]
           [module (find-module module-name)])
      (if (not module)
        (begin
          (warn "unknown module during reconstructing procedure type: ~a\n"
                module-name)
          (compute-procedure-type proc)) ;; fallback
        ($ construct-type <^>
           $ map (^e (if (or (memq e '(* ->)))
                       e
                       (or (%type-name->type module e)
                           (error "unknown type in procedure type info:" e))))
           (vector->list encoded-type 2))))
    (compute-procedure-type proc)))

;; Internal API - called from procedure-type (libproc)
;; Compute #<^ ...> type from the information available in the procedure.
;; Once this info is computed, it is cached in PROC.
(define (compute-procedure-type proc)
  (define (%procedure-type proc)
    (if-let1 clinfo (case-lambda-decompose proc)
      (construct-type </> (map (^v (%procedure-type (caddr v))) clinfo))
      (or (and-let* ([ (closure? proc) ]
                     [code (closure-code proc)])
            ((with-module gauche.internal compiled-code-type) code))
          ;; Fallback - if we don't have detailed type info, just use
          ;; # of arguments.
          (let1 top (%class->proxy <top>)
            (construct-procedure-type (make-list (~ proc'required) top)
                                      (~ proc'optional)
                                      '*)))))
  (define (%method-type meth)
    (construct-procedure-type (map %class->proxy (~ meth'specializers))
                              (~ meth'optional)
                              '*))
  (define (%generic-type gf)
    (construct-type </> (map %method-type (~ gf'methods))))

  (cond [(is-a? proc <procedure>) (%procedure-type proc)]
        [(is-a? proc <generic>)   (%generic-type proc)]
        [(is-a? proc <method>)    (%method-type proc)]
        ;; Dealing with applicable objects are debatable.  Any object can
        ;; become applicable, which makes its type
        ;; (</> <original-type> <type-when-applied>).  That makes type
        ;; operations extremely complicated.
        [else #f]))

(define-cproc %class->proxy (klass::<class>)
  (let* ([ms (-> klass modules)]
         [n (-> klass name)])
    (dolist [m ms]
      (let* ([g::ScmGloc* (Scm_FindBinding (SCM_MODULE m) (SCM_SYMBOL n) 0)])
        (when (!= g NULL)
          (let* ([id (Scm_MakeIdentifier n (SCM_MODULE m) SCM_NIL)])
            (return (Scm_MakeProxyType (SCM_IDENTIFIER id) g))))))
    ;; If we're here, the class doesn't have a known global binding.
    ;; It is possible---a class can be created procedurally at runtime---
    ;; but to be used in a type expession, it must be recognized by the
    ;; compiler, which requires the class is statically bound to a global
    ;; identifier.  We raise an error if that's not the case.
    (Scm_Error "Class %S doesn't have a known global binding and can't be used \
                in a type expression." klass)))

;; Utility.  Called from compiler, too.
(define (construct-procedure-type argtypes ; (<List> <type>)
                                  has-optional? ; <boolean>
                                  rettypes) ; (<List> <type>) or '*
  (let1 rts (if (eq? rettypes '*) '(*) rettypes)
    ($ construct-type <^>
       (if has-optional?
         `(,@argtypes * -> ,@rts)
         `(,@argtypes -> ,@rts)))))

(define-class <^> (<descriptive-type>)
  ((arguments :init-keyword :arguments)    ; <Tuple>
   (results   :init-keyword :results))     ; <Tuple>
  :metaclass <type-constructor-meta>
  :initializer init-^
  :deconstructor deconstruct-^
  :validator validate-^
  :subtype? subtype-^)

;;;
;;; Class: </>
;;;   Creates a union type.
;;;

(define-class </> (<descriptive-type>)
  ((members :init-keyword :members))
  :metaclass <type-constructor-meta>
  :initializer (^[type args]
                 (assume (every (cut is-a? <> <type>) args))
                 (slot-set! type 'name (make-compound-type-name '/ args))
                 (slot-set! type 'members args))
  :deconstructor (^[type] (~ type'members))
  :validator (^[type obj] (any (cut of-type? obj <>) (~ type'members)))
  :subtype? (^[type super] (every (cut subtype? <> super) (~ type'members)))
  :supertype? (^[type sub] (any (cut subtype? sub <>) (~ type'members))))

;;;
;;; Class: <?>
;;;   Creates a boolean-optional type, that is, <type> or #f.
;;;

(define-class <?> (<descriptive-type>)
  ((primary-type :init-keyword :primary-type))
  :metaclass <type-constructor-meta>
  :initializer (^[type args]
                 (let1 ptype (car args)
                   (assume (is-a? ptype <type>))
                   (slot-set! type 'name (make-compound-type-name '? `(,ptype)))
                   (slot-set! type 'primary-type ptype)))
  :deconstructor (^[type] (list (~ type'primary-type)))
  :validator (^[type obj]
               (or (eqv? obj #f) (of-type? obj (~ type'primary-type))))
  :subtype? (^[type super]
              (if (is-a? super <?>)
                (subtype? (~ type'primary-type) (~ super'primary-type))
                (and (of-type? #f super)
                     (subtype? (~ type'primary-type) super))))
  :supertype? (^[type sub] (subtype? sub (~ type'primary-type))))

;;;
;;; Class: <Tuple>
;;;   Fixed-length list, each element having its own type constraints.
;;;

;; (<Tuple> type ... [*])

(define (init-Tuple type args)
  (receive (types rest?) (if (and (pair? args) (eqv? (last args) '*))
                           (values (drop-right args 1) #t)
                           (values args #f))
    (dolist [t types]
      (unless (is-a? t <type>)
        (error "Non-type parameter in <Tuple> constructor:" t)))
    (slot-set! type 'name (make-compound-type-name 'Tuple args))
    (slot-set! type 'elements types)
    (slot-set! type 'allow-rest? rest?)))

(define (deconstruct-Tuple type)
  (if (~ type'allow-rest?)
    (append (~ type'elements) '(*))
    (~ type'elements)))

(define (validate-Tuple type obj)
  (let loop ([obj obj] [elts (~ type'elements)])
    (cond [(null? obj) (null? elts)]
          [(not (pair? obj)) #f]
          [(null? elts) (~ type'allow-rest?)]
          [else (and (of-type? (car obj) (car elts))
                     (loop (cdr obj) (cdr elts)))])))

(define (subtype-Tuple type super)
  (or (eqv? super <list>)
      (and (is-a? super <Tuple>)
           (if (~ type'allow-rest?)
             (<= (length (~ type'elements)) (length (~ super'elements)))
             (= (length (~ type'elements)) (length (~ super'elements))))
           (every (cut subtype? <> <>) (~ type'elements) (~ super'elements)))
      (and (is-a? super <List>)
           (every (cute subtype? <> (~ super'element-type)) (~ type'elements))
           (<= (or (~ super'min-length) 0)
               (length (~ type'elements))
               (or (~ super'max-length) +inf.0)))
      'super))

(define-class <Tuple> (<descriptive-type>)
  ((elements    :init-keyword :elements)
   (allow-rest? :init-keyword :allow-rest?))
  :metaclass <type-constructor-meta>
  :initializer init-Tuple
  :deconstructor deconstruct-Tuple
  :validator  validate-Tuple
  :subtype? subtype-Tuple)

;;;
;;; Class: <List>
;;; Class: <Vector>
;;;   A list or vector of specified types.
;;;

(define (make-init-Seq name)
  (^[type args]
    (apply (^[etype :optional (min #f) (max #f)]
             (unless (or (not min) (real? min))
               (error "min argument must be a real number or #f, but got:" min))
             (unless (or (not max) (real? max))
               (error "max argument must be a real number or #f, but got:" max))
             (slot-set! type 'name
                        (make-min-max-len-type-name name (list etype) min max))
             (slot-set! type 'element-type etype)
             (slot-set! type 'min-length min)
             (slot-set! type 'max-length max))
           args)))

(define (deconstruct-Seq type)
  (list (~ type'element-type) (~ type'min-length) (~ type'max-length)))

(define (validate-List type obj)
  (let ([et (~ type'element-type)]
        [mi (~ type'min-length)]
        [ma (~ type'max-length)])
    (if (not (or mi ma))
      ;; simple case
      (let loop ([obj obj])
        (cond [(null? obj) #t]
              [(not (pair? obj)) #f]
              [(of-type? (car obj) et) (loop (cdr obj))]
              [else #f]))
      ;; general case
      (let loop ([obj obj] [n 0])
        (cond [(null? obj) (or (not mi) (<= mi n))]
              [(and ma (<= ma n)) #f]
              [(not (pair? obj)) #f]
              [(of-type? (car obj) et) (loop (cdr obj) (+ n 1))]
              [else #f])))))

(define (validate-Vector type obj)
  (and (vector? obj)
       (let ([et (~ type'element-type)]
             [mi (~ type'min-length)]
             [ma (~ type'max-length)]
             [len (vector-length obj)])
         (and (or (not mi) (<= mi len))
              (or (not ma) (<= len ma))
              (let loop ([i 0])
                (cond [(= i len) #t]
                      [(of-type? (vector-ref obj i) et) (loop (+ i 1))]
                      [else #f]))))))

(define (make-subtype-Seq base-type)
  (^[type super]
    (or (eqv? super base-type)
        (and (is-a? super (class-of type))
             (subtype? (~ type'element-type) (~ super'element-type))
             (>= (or (~ type'min-length) 0)
                 (or (~ super'min-length) 0))
             (<= (or (~ type'max-length) +inf.0)
                 (or (~ super'max-length) +inf.0)))
        'super)))

(define-class <List> (<descriptive-type>)
  ((element-type :init-keyword :element-type)
   (min-length :init-keyword :min-length :init-value #f)
   (max-length :init-keyword :max-length :init-value #f))
  :metaclass <type-constructor-meta>
  :initializer (make-init-Seq 'List)
  :deconstructor deconstruct-Seq
  :validator validate-List
  :subtype? (make-subtype-Seq <list>))

(define-class <Vector> (<descriptive-type>)
  ((element-type :init-keyword :element-type)
   (min-length :init-keyword :min-length :init-value #f)
   (max-length :init-keyword :max-length :init-value #f))
  :metaclass <type-constructor-meta>
  :initializer (make-init-Seq 'Vector)
  :deconstructor deconstruct-Seq
  :validator validate-Vector
  :subtype? (make-subtype-Seq <vector>))

;;;
;;; Class: <Assortment>
;;;   A type consists of a set of concrete objects.
;;;   If it has only one member, you can think of it as a singleton type.
;;;   If it has multiple members, it's a union of singleton types.
;;;   The members (instances) are sorted by default-comparator so that
;;;   equivalence is easily tested.
;;;

(define-class <Assortment> (<descriptive-type>)
  ((instances :init-keyword :instance))
  :metaclass <type-constructor-meta>
  :initializer (^[type args]
                 (define objs (map unwrap-syntax args))
                 (slot-set! type 'name
                            (string->symbol
                             (string-append
                              "<Assortment " (x->string objs) ">")))
                 (slot-set! type 'instances (sort objs)))
  :deconstructor (^[type] (list (~ type'instances)))
  :validator (^[type obj] (boolean (memv obj (~ type'instances))))
  :subtype? (^[type super] (every (^[obj] (of-type? obj super))
                                  (~ type'instances)))
  :supertype? (^[type sub] #f))

;;;
;;; <native-type> : Types for bridging Scheme and C
;;;

;; Each of these types has a corresponding cgen-type that maintains
;; the knowledge how it is represented in C.

(inline-stub
 (define-cfn make_native_type (name::(const char*)
                               super
                               c-type-name::(const char*)
                               size::size_t
                               alignment::size_t
                               c-of-type::(.function (obj)::int *)
                               c-ref::(.function (ptr::void*)::ScmObj *)
                               c-set::(.function (ptr::void* obj)::void *))
   :static
   (let* ([z::ScmNativeType*
           (SCM_NEW_INSTANCE ScmNativeType (& Scm_NativeTypeClass))])
     (set! (-> z name) (SCM_INTERN name))
     (set! (-> z super) super)
     (set! (-> z c-type-name) c-type-name)
     (set! (-> z c-of-type) c-of-type)
     (set! (-> z c-ref) c-ref)
     (set! (-> z c-set) c-set)
     (set! (-> z size) size)
     (set! (-> z alignment) alignment)
     (return (SCM_OBJ z))))

 ;; Primitive native type name -> native type instance
 (define-cvar builtin-native-types :static)

 (initcode
  (set! builtin-native-types (Scm_MakeHashTableSimple SCM_HASH_EQ 16)))

 (define-cproc %builtin-native-type-lookup (name::<symbol>)
    (return (Scm_HashTableRef (SCM_HASH_TABLE builtin-native-types)
                              (SCM_OBJ name)
                              SCM_FALSE)))

 ;; define-native-type NAME SUPER CTYPE PRED BOX UNBOX
 ;;   NAME - Symbol for Scheme name
 ;;   SUPER - C macro name for superclass
 ;;   CTYPE - C type name
 ;;   PRED - C function ScmObj -> int for type predicate
 ;;   BOX - C function ctype -> ScmObj for boxing
 ;;   UNBOX - C function ScmObj -> ctype for unboxing

 (define-cise-stmt define-native-type
   [(_ name super ctype pred box unbox)
    (define c-ref-name (symbol-append name '-ptr-ref))
    (define c-set-name (symbol-append name '-ptr-set))

    (cgen-decl
     (cise-render-to-string
      `(define-cfn ,c-ref-name (ptr::void*) :static
         (let* ([pp :: (,ctype *) (cast (,ctype *) ptr)])
           (return (,box (* pp)))))
      'toplevel))
    (cgen-decl
     (cise-render-to-string
      `(define-cfn ,c-set-name (ptr::void* obj) ::void :static
         (let* ([pp :: (,ctype *) (cast (,ctype *) ptr)])
           (unless (,pred obj)
             (SCM_TYPE_ERROR obj ,(x->string name)))
           (set! (* pp) (,unbox obj))))
      'toplevel))
    `(let* ([z (make_native_type ,(symbol->string name)
                                 (SCM_OBJ ,super) ,(x->string ctype)
                                 (sizeof (.type ,ctype))
                                 (SCM_ALIGNOF (.type ,ctype))
                                 ,pred
                                 ,c-ref-name
                                 ,c-set-name)])
       (Scm_HashTableSet (SCM_HASH_TABLE builtin-native-types)
                         ',ctype (SCM_OBJ z) 0)
       (Scm_MakeBinding (Scm_GaucheModule)
                        (SCM_SYMBOL (-> (SCM_NATIVE_TYPE z) name)) z
                        SCM_BINDING_INLINABLE))])

 (define-cfn native_fixnumP (obj) ::int :static
   (return (SCM_INTP obj)))

 ;; NB: Range check is also in ext/uvector/uvector.scm.  May be integrated.
 (define-cfn native_s8P (obj) ::int :static
   (return (and (SCM_INTP obj)
                (>= (SCM_INT_VALUE obj) -128)
                (<= (SCM_INT_VALUE obj) 127))))
 (define-cfn native_u8P (obj) ::int :static
   (return (and (SCM_INTP obj)
                (>= (SCM_INT_VALUE obj) 0)
                (<= (SCM_INT_VALUE obj) 255))))
 (define-cfn native_s16P (obj) ::int :static
   (return (and (SCM_INTP obj)
                (>= (SCM_INT_VALUE obj) -32768)
                (<= (SCM_INT_VALUE obj) 32767))))
 (define-cfn native_u16P (obj) ::int :static
   (return (and (SCM_INTP obj)
                (>= (SCM_INT_VALUE obj) 0)
                (<= (SCM_INT_VALUE obj) 65535))))
 (define-cfn native_s32P (obj) ::int :static
   (.if (== SIZEOF_LONG 4)
        (return (or (SCM_INTP obj)
                    (and (SCM_BIGNUMP obj)
                         (>= (Scm_NumCmp obj '#x-8000_0000) 0)
                         (<= (Scm_NumCmp obj '#x7fff_ffff) 0))))
        (return (and (SCM_INTP obj)
                     (>= (SCM_INT_VALUE obj) #x-8000_0000)
                     (<= (SCM_INT_VALUE obj) #x7fff_ffff)))))
 (define-cfn native_u32P (obj) ::int :static
   (.if (== SIZEOF_LONG 4)
        (return (or (and (SCM_INTP obj)
                         (>= (SCM_INT_VALUE obj) 0))
                    (and (SCM_BIGNUMP obj)
                         (>= (Scm_Sign obj) 0)
                         (<= (Scm_NumCmp obj '#xffff_ffff) 0))))
        (return (and (SCM_INTP obj)
                     (>= (SCM_INT_VALUE obj) 0)
                     (<= (SCM_INT_VALUE obj) #xffff_ffff)))))
 (define-cfn native_s64P (obj) ::int :static
   (return (or (SCM_INTP obj)
               (and (SCM_BIGNUMP obj)
                    (>= (Scm_NumCmp obj '#x-8000_0000_0000_0000) 0)
                    (<= (Scm_NumCmp obj '#x7fff_ffff_ffff_ffff) 0)))))
 (define-cfn native_u64P (obj) ::int :static
   (return (or (and (SCM_INTP obj)
                    (>= (SCM_INT_VALUE obj) 0))
               (and (SCM_BIGNUMP obj)
                    (>= (Scm_Sign obj) 0)
                    (<= (Scm_NumCmp obj '#xffff_ffff_ffff_ffff) 0)))))

 (define-cfn native_shortP (obj) ::int :static
   (return (and (SCM_INTP obj)
                (>= (SCM_INT_VALUE obj) SHRT_MIN)
                (<= (SCM_INT_VALUE obj) SHRT_MAX))))
 (define-cfn native_ushortP (obj) ::int :static
   (return (and (SCM_INTP obj)
                (>= (SCM_INT_VALUE obj) 0)
                (<= (SCM_INT_VALUE obj) USHRT_MAX))))
 (define-cfn native_intP (obj) ::int :static
   (.if (== SIZEOF_LONG 4)
        (if (SCM_BIGNUMP obj)
          (let* ([oor::int FALSE]
                 [v::long (Scm_GetIntegerClamp obj SCM_CLAMP_BOTH (& oor))])
            (return (and (not oor)
                         (>= v INT_MIN)
                         (<= v INT_MAX))))
          (return (SCM_INTP obj)))
        (if (SCM_INTP obj)
          (let* ([v::long (SCM_INT_VALUE obj)])
            (return (and (>= v INT_MIN)
                         (<= v INT_MAX))))
          (return FALSE))))
 (define-cfn native_uintP (obj) ::int :static
   (.if (== SIZEOF_LONG 4)
        (if (SCM_BIGNUMP obj)
          (let* ([oor::int FALSE])
            (cast (void) (Scm_GetIntegerUClamp obj SCM_CLAMP_BOTH (& oor)))
            (return (not oor)))
          (return (and (SCM_INTP obj) (>= (SCM_INT_VALUE obj) 0))))
        (return (and (SCM_INTP obj)
                     (>= (SCM_INT_VALUE obj) 0)
                     (<= (SCM_INT_VALUE obj) UINT_MAX)))))
 (define-cfn native_longP (obj) ::int :static
   (if (SCM_BIGNUMP obj)
     (let* ([oor::int FALSE])
       (cast void (Scm_GetIntegerClamp obj SCM_CLAMP_BOTH (& oor)))
       (return (not oor)))
     (return (SCM_INTP obj))))
 (define-cfn native_ulongP (obj) ::int :static
   (if (SCM_BIGNUMP obj)
     (let* ([oor::int FALSE])
       (cast void (Scm_GetIntegerUClamp obj SCM_CLAMP_BOTH (& oor)))
       (return (not oor)))
     (return (and (SCM_INTP obj) (>= (SCM_INT_VALUE obj) 0)))))

 ;; we don't range-check flonums
 (define-cfn native_realP (obj) ::int :static
   (return (SCM_REALP obj)))

 ;; In C, char used to be used for both 'character' and 'one byte integer'.
 ;; Four our purpose, we map native char type to our char (in 1 byte range).
 ;; Foreign functions that uses char as one byte integer, we can use <int8>
 ;; or <uint8>.
 (define-cfn native_charP (obj) ::int :static
   (return (and (SCM_CHARP obj)
                (<= (SCM_CHAR_VALUE obj) 255))))

 (define-cfn native_cstrP (obj) ::int :static
   (return (SCM_STRINGP obj)))
 (define-cfn get_cstr (obj) ::(const char*) :static
   (SCM_ASSERT (SCM_STRINGP obj))
   (return (Scm_GetStringConst (SCM_STRING obj))))

 ;; subrs returning <void> actually return #<undef>
 (define-cfn native_voidP (obj) ::int :static
   (return (SCM_UNDEFINEDP obj)))

 (define-cfn native_iportP (obj) ::int :static
   (return (SCM_IPORTP obj)))
 (define-cfn native_oportP (obj) ::int :static
   (return (SCM_OPORTP obj)))
 (define-cfn native_closureP (obj) ::int :static
   (return (SCM_CLOSUREP obj)))

 (initcode
  (define-native-type <fixnum>  SCM_CLASS_INTEGER ScmSmallInt native_fixnumP
    SCM_MAKE_INT SCM_INT_VALUE)
  (define-native-type <short>   SCM_CLASS_INTEGER short native_shortP
    SCM_MAKE_INT SCM_INT_VALUE)
  (define-native-type <ushort>  SCM_CLASS_INTEGER u_short native_ushortP
    SCM_MAKE_INT SCM_INT_VALUE)
  (define-native-type <int>     SCM_CLASS_INTEGER int native_intP
    Scm_MakeInteger Scm_GetInteger)
  (define-native-type <uint>    SCM_CLASS_INTEGER u_int native_uintP
    Scm_MakeIntegerU Scm_GetIntegerU)
  (define-native-type <long>    SCM_CLASS_INTEGER long native_longP
    Scm_MakeInteger Scm_GetInteger)
  (define-native-type <ulong>   SCM_CLASS_INTEGER u_long native_ulongP
    Scm_MakeIntegerU Scm_GetIntegerU)
  (define-native-type <int8>    SCM_CLASS_INTEGER int8_t native_s8P
    SCM_MAKE_INT SCM_INT_VALUE)
  (define-native-type <uint8>   SCM_CLASS_INTEGER uint8_t native_u8P
    SCM_MAKE_INT SCM_INT_VALUE)
  (define-native-type <int16>   SCM_CLASS_INTEGER int16_t native_s16P
    SCM_MAKE_INT SCM_INT_VALUE)
  (define-native-type <uint16>  SCM_CLASS_INTEGER uint16_t native_u16P
    SCM_MAKE_INT SCM_INT_VALUE)
  (define-native-type <int32>   SCM_CLASS_INTEGER int32_t native_s32P
    Scm_MakeInteger Scm_GetInteger)
  (define-native-type <uint32>  SCM_CLASS_INTEGER uint32_t native_u32P
    Scm_MakeIntegerU Scm_GetIntegerU)
  (define-native-type <int64>   SCM_CLASS_INTEGER int64_t native_s64P
    Scm_MakeInteger64 Scm_GetInteger64)
  (define-native-type <uint64>  SCM_CLASS_INTEGER uint64_t native_u64P
    Scm_MakeIntegerU64 Scm_GetIntegerU64)

  ;; We use <native-char>, for <char> is already used for Scheme characters.
  (define-native-type <native-char> SCM_CLASS_INTEGER char native_charP
    SCM_MAKE_CHAR SCM_CHAR_VALUE)

  (define-native-type <size_t>  SCM_CLASS_INTEGER size_t Scm_IntegerFitsSizeP
    Scm_SizeToInteger Scm_IntegerToSize)
  (define-native-type <ssize_t> SCM_CLASS_INTEGER ssize_t Scm_IntegerFitsSsizeP
    Scm_SsizeToInteger Scm_IntegerToSsize)
  (define-native-type <ptrdiff_t> SCM_CLASS_INTEGER ptrdiff_t Scm_IntegerFitsPtrdiffP
    Scm_PtrdiffToInteger Scm_IntegerToPtrdiff)
  (define-native-type <off_t> SCM_CLASS_INTEGER off_t Scm_IntegerFitsOffsetP
    Scm_OffsetToInteger Scm_IntegerToOffset)
  (define-native-type <intptr_t> SCM_CLASS_INTEGER intptr_t Scm_IntegerFitsIntptrP
    Scm_IntptrToInteger Scm_IntegerToIntptr)
  (define-native-type <uintptr_t> SCM_CLASS_INTEGER uintptr_t Scm_IntegerFitsUintptrP
    Scm_UintptrToInteger Scm_IntegerToUintptr)

  (define-native-type <float>   SCM_CLASS_REAL float native_realP
    Scm_MakeFlonum Scm_GetDouble)
  (define-native-type <double>  SCM_CLASS_REAL double native_realP
    Scm_MakeFlonum Scm_GetDouble)

  (define-native-type <const-cstring> SCM_CLASS_STRING "const char*" native_cstrP
    SCM_MAKE_STR_COPYING get_cstr)
  (define-native-type <input-port>  SCM_CLASS_PORT ScmPort* native_iportP
    SCM_OBJ SCM_PORT)
  (define-native-type <output-port> SCM_CLASS_PORT ScmPort* native_oportP
    SCM_OBJ SCM_PORT)
  (define-native-type <closure> SCM_CLASS_PROCEDURE ScmClosure* native_closureP
    SCM_OBJ SCM_CLOSURE)

  ;; <void> needs special care, as it doesn't have a real C type.
  (let* ([z (make_native_type "<void>" (SCM_OBJ SCM_CLASS_TOP) "void"
                              0 1 native_voidP NULL NULL)])
    (Scm_HashTableSet (SCM_HASH_TABLE builtin-native-types)
                      'void (SCM_OBJ z) 0)
    (Scm_MakeBinding (Scm_GaucheModule) (SCM_SYMBOL '<void>) z
                     SCM_BINDING_INLINABLE))
  ))

;;
;; Native handle
;;
;;  The struct definition is in priv/typeP.h

(inline-stub
 (define-cclass <native-handle> :base :no-meta
   "ScmNativeHandle*" "Scm_NativeHandleClass"
   (c "SCM_CLASS_DEFAULT_CPL")
   ((name)
    (type :type <native-type>))
   (printer (Scm_Printf port "#<native-handle %A@%p>"
                        (-> (SCM_NATIVE_HANDLE obj) name)
                        (-> (SCM_NATIVE_HANDLE obj) ptr))))

 (define-cfn Scm__MakeNativeHandle (ptr::void*
                                    type::ScmNativeType*
                                    name::ScmObj
                                    region-min::void*
                                    region-max::void*
                                    owner::ScmObj
                                    attrs::ScmObj
                                    flags::u_long)
   (let* ([h::ScmNativeHandle* (SCM_NEW ScmNativeHandle)])
     (SCM_SET_CLASS h SCM_CLASS_NATIVE_HANDLE)
     (set! (-> h ptr) ptr
           (-> h type) type
           (-> h name) name
           (-> h region-min) region-min
           (-> h region-max) region-max
           (-> h owner) owner
           (-> h attrs) attrs
           (-> h flags) flags)
     (return (SCM_OBJ h))))
 )

;;
;; Native composite types
;;

(inline-stub
 (define-cfn native_handleP (obj) ::int :static
   (return (SCM_NATIVE_HANDLE_P obj)))

 ;; Helper function to initialize common fields of composite native types
 (define-cfn init-native-type-common (nt::ScmNativeType*
                                      name::(const char*)
                                      super::ScmObj
                                      c-type-name::(const char*)
                                      size::size_t
                                      alignment::size_t
                                      c-of-type::(.function (obj)::int *)
                                      c-ref::(.function (ptr::void*)::ScmObj *)
                                      c-set::(.function (ptr::void* obj)::void *))
   ::void :static
   (set! (-> nt name) (SCM_INTERN name))
   (set! (-> nt super) super)
   (set! (-> nt c-type-name) c-type-name)
   (set! (-> nt c-of-type) c-of-type)
   (set! (-> nt c-ref) c-ref)
   (set! (-> nt c-set) c-set)
   (set! (-> nt size) size)
   (set! (-> nt alignment) alignment))
 )

(define-cproc %make-pointer-type (pointer-type-name::<const-cstring>
                                  pointee-type)
  (let* ([z::ScmNativePointer*
          (SCM_NEW_INSTANCE ScmNativePointer (& Scm_NativePointerClass))])
    ;; Fill in common fields
    (init-native-type-common (& (-> z common))
                             pointer-type-name
                             (SCM_OBJ SCM_CLASS_TOP)
                             "ScmNativeHandle*"
                             (sizeof (.type void*))
                             (SCM_ALIGNOF (.type void*))
                             native_handleP
                             NULL
                             NULL)
    ;; Fill in type-specific fields
    (SCM_ASSERT (SCM_NATIVE_TYPE_P pointee-type))
    (set! (-> z pointee_type) (SCM_NATIVE_TYPE pointee-type))
    (return (SCM_OBJ z))))

(define (make-pointer-type pointee-type)
  (assume-type pointee-type <native-type>)
  (let* ([bare-name (regexp-replace* (symbol->string (~ pointee-type'name))
                                     #/^</ ""
                                     #/>$/ "")]
         [pointer-name #"<~|bare-name|*>"])
    (%make-pointer-type pointer-name pointee-type)))

(define-cproc %make-native-function-type (type-name::<const-cstring>
                                          return-type
                                          argument-types
                                          varargs?::<boolean>)
  (let* ([z::ScmNativeFunction*
          (SCM_NEW_INSTANCE ScmNativeFunction (& Scm_NativeFunctionClass))])
    ;; Fill in common fields
    (init-native-type-common (& (-> z common))
                             type-name
                             (SCM_OBJ SCM_CLASS_TOP)
                             "ScmNativeHandle*"
                             (sizeof (.type void*))
                             (SCM_ALIGNOF (.type void*))
                             native_handleP
                             NULL
                             NULL)
    ;; Fill in type-specific fields
    (SCM_ASSERT (SCM_NATIVE_TYPE_P return-type))
    (set! (-> z return_type) (SCM_NATIVE_TYPE return-type))
    (set! (-> z arg_types) argument-types)
    (set! (-> z varargs) (?: varargs? 1 0))
    (return (SCM_OBJ z))))

;; Argument-types are list of native types, optionally end with
;; a symbol ... for varargs.
(define (make-native-function-type return-type
                                   argument-types)
  (assume-type return-type <native-type>)
  (receive (arg-types vararg?)
      (if (and (pair? argument-types) (eq? (last argument-types) '...))
        (values (drop-right argument-types 1) #t)
        (values argument-types #f))
    (dolist [arg-type arg-types]
      (assume-type arg-type <native-type>))
    (%make-native-function-type
     (format "~{ ~a~}~:[~; ...~] -> ~a"
             (map (cut ~ <>'name) arg-types) vararg?
             (~ return-type'name))
     return-type arg-types vararg?)))

;; For array, we keep element-type and dimensions in dedicated fields.
;; Each <dim> is a nonnegative fixnum.  The first <dim> can be -1,
;; indicating it is not specified (C allows it).
(define-cproc %make-native-array-type (type-name::<const-cstring>
                                       element-type
                                       size::<fixnum>
                                       alignment::<fixnum>
                                       dimensions)
  (let* ([z::ScmNativeArray*
          (SCM_NEW_INSTANCE ScmNativeArray (& Scm_NativeArrayClass))])
    ;; Fill in common fields
    (init-native-type-common (& (-> z common))
                             type-name
                             (SCM_OBJ SCM_CLASS_TOP)
                             "ScmNativeHandle*"
                             size
                             alignment
                             native_handleP
                             NULL
                             NULL)
    ;; Fill in type-specific fields
    (SCM_ASSERT (SCM_NATIVE_TYPE_P element-type))
    (set! (-> z element_type) (SCM_NATIVE_TYPE element-type))
    (set! (-> z dimensions) dimensions)
    (return (SCM_OBJ z))))

(define (make-native-array-type element-type dimensions)
  (assume-type element-type <native-type>)
  (let loop ([dims dimensions])
    (cond [(null? dims)]
          [(not (pair? dims))
           (error "Bad native array dimensions; must be a proper list, but got:"
                  dimensions)]
          [(and (eq? dims dimensions)
                (eq? (car dims) '*))
           (loop (cdr dims))]   ;first dimension can be *
          [(and (exact-integer? (car dims))
                (>= (car dims) 0))
           (loop (cdr dims))]
          [else
           (error "Bad native array dimensions; must be a list of nonnegative \
                   integers, or '* at the last position, but got:"
                  dimensions)]))
  (let ([name (format "~a~a" (~ element-type 'name) dimensions)]
        [num-elts (if (eq? (car dimensions) '*)
                    0                   ; unknown sized array
                    (fold * 1 dimensions))]
        [elt-size (~ element-type'size)])
    (%make-native-array-type name element-type
                             (* elt-size num-elts)
                             (~ element-type'alignment)
                             dimensions)))

;; For struct/union, we keep tag and field-list in dedicated fields.
(define-cproc %make-native-struct/union-type (klass::<class>
                                              type-name::<const-cstring>
                                              size::<fixnum>
                                              alignment::<fixnum>
                                              tag-name::<symbol>?
                                              field-list)
  (let* ([z::ScmNativeStruct* (SCM_NEW_INSTANCE ScmNativeStruct klass)])
    ;; Fill in common fields
    (init-native-type-common (& (-> z common))
                             type-name
                             (SCM_OBJ SCM_CLASS_TOP)
                             "ScmNativeHandle*"
                             size
                             alignment
                             native_handleP
                             NULL
                             NULL)
    ;; Fill in type-specific fields
    (set! (-> z tag) (SCM_OBJ tag-name))
    (set! (-> z fields) field-list)
    (return (SCM_OBJ z))))

(define (struct-size-roundup size alignment)
  (* alignment (quotient (+ size alignment -1) alignment)))

(define (make-native-struct/union-type tag fields struct?)
  (let loop ([fs fields] [offset 0] [alignment 1] [descs '()])
    (match fs
      [()
       (let* ([size (struct-size-roundup offset alignment)]
              [name (x->string tag)])
         (%make-native-struct/union-type
          (if struct? <native-struct> <native-union>)
          name size alignment tag (reverse descs)))]
      [(((? symbol? fname) ftype) . rest)
       (assume-type ftype <native-type>)
       (let* ([falign (~ ftype'alignment)]
              [foffset (if struct?
                         (struct-size-roundup offset falign)
                         0)]
              [next (if struct?
                      (+ foffset (~ ftype'size))
                      (max offset (~ ftype'size)))]
              [new-align (max alignment falign)])
         (loop rest
               next
               new-align
               (cons (list fname ftype foffset) descs)))]
      [_
       (error "Bad native struct fields; must be a proper list, but got:"
              fields)])))

(define (make-native-struct-type tag fields)
  (make-native-struct/union-type tag fields #t))

(define (make-native-union-type tag fields)
  (make-native-struct/union-type tag fields #f))

;;;
;;; Make exported symbol visible from outside
;;;

(let ((xfer (with-module gauche.internal %transfer-bindings)))
  (xfer (current-module)
        (find-module 'gauche)
        '(<type-constructor-meta>
          <descriptive-type>
          <native-type> <native-pointer> <native-function>
          <native-array> <native-struct> <native-union>
          <^> </> <?> <Tuple> <List> <Vector> <Assortment>
          type? subtype? of-type?))
  (xfer (current-module)
        (find-module 'gauche.internal)
        '(construct-type
          deconstruct-type
          wrap-with-proxy-type
          proxy-type-ref
          proxy-type-id
          ;; followings are called from procedure-type (libproc)
          reconstruct-procedure-type
          compute-procedure-type
          construct-procedure-type)))
